// **************************************************************************
// *				 Sporting Clays for the Playstation						*
// *					   (C) 1999 Charles Doty                          	*
// *	 	   3D Scene Source file contains 3D related code				*
// **************************************************************************

#include "scene3d.h"

#define ZSTART      200                     // Start of the Z buffer

Object3D Objects[MAXOBJECTS];

DWORD Scene3DInitialize()
{
	DWORD 	loop;

    slPerspective(DEGtoANG(40));
    slZdspLevel(3);

	for (loop = 0; loop < MAXOBJECTS; loop++)
	{
		Objects[loop].Used 			= FALSE;
		Objects[loop].CurrentFrame	= 0;
	}

	return ERR_NOERROR;
}

DWORD AddObject(XPDATA *object, WORD *ObjectID, POS *position, ROT *rotation,
    SCALE *scale)
{
    DWORD       loop    = 0;
	long		color;

    if (NULL == object || NULL == ObjectID || NULL == position || NULL == rotation || NULL == scale)
    {
#ifdef DEBUG
        slPrint("Bad Parameters to AddObject\n", slLocate(9, 2));
#endif
        *ObjectID   = 0xFFFF;
        return ERR_BADPARAMS;
    }

    for (loop = 0; loop < MAXOBJECTS; loop++)
    {
    	if (FALSE == Objects[loop].Used)
    	{
    		break;
    	}
    }

    if (loop >= MAXOBJECTS)
    {
#ifdef DEBUG
        slPrint("No object available in AddObject\n", slLocate(9, 2));
#endif

    	return ERR_NONEAVAILABLE;
    }

    Objects[loop].Position[X]               = toFIXED(position->x);
    Objects[loop].Position[Y]               = toFIXED(position->y);
    Objects[loop].Position[Z]               = toFIXED(position->z + ZSTART);
    Objects[loop].Rotation[X]               = DEGtoANG(rotation->x);
    Objects[loop].Rotation[Y]               = DEGtoANG(rotation->y);
    Objects[loop].Rotation[Z]               = DEGtoANG(rotation->z);
    Objects[loop].Scale[X]                  = toFIXED(scale->x);
    Objects[loop].Scale[Y]                  = toFIXED(scale->y);
    Objects[loop].Scale[Z]                  = toFIXED(scale->z);
	Objects[loop].CurrentFrame				= 0;
    Objects[loop].Used 						= TRUE;
    Objects[loop].Object[0].Primatives      = object;

	*ObjectID = loop;

	return ERR_NOERROR;
}

DWORD AddAnimation(XPDATA *object, WORD ObjectID, long frame)
{
    if (NULL == object)
    {
#ifdef DEBUG
        slPrint("Bad Parameters to AddAnimation\n", slLocate(9, 2));
#endif

    	return ERR_BADPARAMS;
    }

    Objects[ObjectID].Object[frame].Primatives      = object;

	return ERR_NOERROR;
}

DWORD RemoveObject(WORD ObjectID)
{
	if (ObjectID > MAXOBJECTS || FALSE == Objects[ObjectID].Used)
	{
#ifdef DEBUG
      slPrint("Bad Parameters to RemoveObject\n", slLocate(9, 2));
#endif

		return ERR_BADPARAMS;
	}

    Objects[ObjectID].Used          = FALSE;
    Objects[ObjectID].CurrentFrame  = 0;

	return ERR_NOERROR;
}

DWORD MoveObject(WORD ObjectID, POS *position)
{
	if (ObjectID > MAXOBJECTS || FALSE == Objects[ObjectID].Used)
	{
#ifdef DEBUG
        slPrint("Bad Parameters to MoveObject\n", slLocate(9, 2));
#endif

		return ERR_BADPARAMS;
	}

    Objects[ObjectID].Position[X]   = toFIXED(position->x);
    Objects[ObjectID].Position[Y]   = toFIXED(position->y);
    Objects[ObjectID].Position[Z]   = toFIXED(position->z + ZSTART);  

    return ERR_NOERROR;
}

DWORD RotateObject(WORD ObjectID, ROT *rotation)
{
	if (ObjectID > MAXOBJECTS || FALSE == Objects[ObjectID].Used)
	{
#ifdef DEBUG
        slPrint("Bad Parameters to RotateObject\n", slLocate(9, 2));
#endif

		return ERR_BADPARAMS;
	}

    Objects[ObjectID].Rotation[X]   = DEGtoANG(rotation->x);
    Objects[ObjectID].Rotation[Y]   = DEGtoANG(rotation->y);
    Objects[ObjectID].Rotation[Z]   = DEGtoANG(rotation->z);

    return ERR_NOERROR;
}

DWORD SetObjectFrame(WORD ObjectID, long frame)
{
	if (ObjectID > MAXOBJECTS || FALSE == Objects[ObjectID].Used)
	{
#ifdef DEBUG
        slPrint("Bad Parameters to SetObjectFrame\n", slLocate(9, 2));
#endif

        return ERR_BADPARAMS;
	}

	Objects[ObjectID].CurrentFrame = frame;

	return ERR_NOERROR;
}

BOOL CheckIntersection(WORD ObjectID, short x, short y, short range, short *distance, short *xdir,
	short *ydir)
{
    Sint32 Pos[XY];

	short x1;
	short y1;
	short x2;
	short y2;
	short xtotal;
	short ytotal;
	short xcenter;
	short ycenter;

	if (ObjectID > MAXOBJECTS || FALSE == Objects[ObjectID].Used || NULL == distance || NULL == xdir
		|| NULL == ydir)
	{
#ifdef DEBUG
        slPrint("Bad Parameters to Check Intersection\n", slLocate(9, 2));
#endif
		return FALSE;
	}

    slConvert3Dto2D(Objects[ObjectID].Position, Pos);

    x1  = Pos[X] - CLAYSIZE;
    y1  = Pos[Y] - CLAYSIZE;
    x2  = Pos[X] + CLAYSIZE;
    y2  = Pos[Y] + CLAYSIZE;

	if (x + range >= x1 && x - range <= x2 && y + range >= y1 && y - range <= y2)
	{
		xcenter = (x1 + x2) / 2;
		xtotal 	= (x - xcenter) ^ 2;
		ycenter = (y1 + y2) / 2;
		ytotal 	= (y - ycenter) ^ 2;

        *distance = (short)slSquart((xtotal + ytotal));
		*xdir	  = x < xcenter ? -1 : 1;
		*ydir	  = y > ycenter ? -1 : 1;

		return TRUE;
	}

	return FALSE;
}

BOOL CheckFloor(WORD ObjectID, short y)
{
    Sint32 Pos[XY];

	if (ObjectID > MAXOBJECTS || FALSE == Objects[ObjectID].Used)
	{
#ifdef DEBUG
        slPrint("Bad Parameters to Check Floor\n", slLocate(9, 2));
#endif
		return FALSE;
	}

    if (slConvert3Dto2D(Objects[ObjectID].Position, Pos) > toFIXED(300)
        && Pos[Y] + SCREENHEIGHT / 2 >= y)
    {
		return TRUE;
	}

	return FALSE;
}

BOOL CheckScreen(WORD ObjectID)
{
	long x1;
	long y1;
	long x2;
	long y2;

	if (ObjectID > MAXOBJECTS || FALSE == Objects[ObjectID].Used)
	{
#ifdef DEBUG
        slPrint("Bad Parameters to Check Screen\n", slLocate(9, 2));
#endif
		return FALSE;
	}

    if (slCheckOnScreen(Objects[ObjectID].Position, toFIXED(CLAYSIZE)) < 0)
    {
		return TRUE;
	}

	return FALSE;
}

DWORD Render()
{
    DWORD       loop;
    DWORD       objects;
    long        temp;
    long        OtZ;
    long        interpolation;
    long        flag;
    long        objx;
    long        objy;
    long		frame;
    short		Obj;
    short		ObjID[MAXOBJECTS];
    long		zpos[MAXOBJECTS];
    long		tmp;
    BOOL		Changed = TRUE;

    for (loop = 0; loop < MAXOBJECTS; loop++)
    {
		if (FALSE == Objects[loop].Used)
		{
			ObjID[loop] = -1;
		}

		else
		{
			ObjID[loop] = loop;
		}

        zpos[loop]  = Objects[loop].Position[Z];
	}

    while (TRUE == Changed)
    {
    	Changed = FALSE;

	    for (loop = 0; loop < (MAXOBJECTS - 1); loop++)
    	{
			if (-1 == ObjID[loop] && ObjID[loop + 1] >= 0)
			{
				ObjID[loop] 	= ObjID[loop + 1];
				ObjID[loop + 1] = -1;
				zpos[loop]		= zpos[loop + 1];

				Changed = TRUE;

				continue;
			}

			if (ObjID[loop] != -1 && ObjID[loop + 1] != -1)
			{
				if (zpos[loop] > zpos[loop + 1])
				{
					tmp 			= ObjID[loop];
					ObjID[loop]		= ObjID[loop + 1];
					ObjID[loop + 1] = tmp;
					tmp 			= zpos[loop];
					zpos[loop]  	= zpos[loop + 1];
					zpos[loop + 1]  = tmp;

					Changed = TRUE;
				}
			}
		}
    }

    for (loop = 0; loop < MAXOBJECTS; loop++)
    {
		Obj	= ObjID[loop];

		if (Obj < 0)
		{
			break;
		}

        frame       = Objects[Obj].CurrentFrame;

		slPushMatrix();

        slTranslate(Objects[Obj].Position[X], Objects[Obj].Position[Y],
            Objects[Obj].Position[Z]);

        slScale(Objects[Obj].Scale[X], Objects[Obj].Scale[Y], Objects[Obj].Scale[Z]);

        slRotX(Objects[Obj].Rotation[X]);
        slRotY(Objects[Obj].Rotation[Y]);
        slRotZ(Objects[Obj].Rotation[Z]);

        slPutPolygon((PDATA *)Objects[Obj].Object[frame].Primatives);

		slPopMatrix();
    }

	return ERR_NOERROR;
}

DWORD LoadTIM(u_long *TIM)
{
//    TIM_IMAGE       image;

	if (NULL == TIM)
	{
#ifdef DEBUG
        slPrint("Bad Parameters to LoadTIM.\n", slLocate(9, 2));
#endif

		return ERR_BADPARAMS;
	}

/*
	if (OpenTIM(TIM) != 0)
	{
#ifdef DEBUG
        slPrint("Unable to open TIM.\n", slLocate(9, 2));
#endif

		return ERR_FILEERROR;
	}

	while (ReadTIM(&image) != 0)
	{
		if (image.caddr != 0)
		{
			LoadImage(image.crect, image.caddr);
		}

		if (image.paddr != 0)
		{
			LoadImage(image.prect, image.paddr);
		}
	}
*/

	return ERR_NOERROR;
}
