// **************************************************************************
// *				 Sporting Clays for the Playstation						*
// *					   (C) 1999 Charles Doty                          	*
// *	 	   3D Scene Source file contains 3D related code				*
// **************************************************************************

#include "scene3d.h"

#define SCREENZ     1024                    // screen projection
#define ZSTART		600						// Start of the Z buffer

Object3D		Objects[MAXOBJECTS];

extern	GsOT    OT[2];                              // Ordering table

DWORD Scene3DInitialize()
{
	DWORD 	loop;

	for (loop = 0; loop < MAXOBJECTS; loop++)
	{
		Objects[loop].Used 			= FALSE;
		Objects[loop].CurrentFrame	= 0;
	}

    SetGeomOffset(SCREENWIDTH / 2, SCREENHEIGHT / 2);
    SetGeomScreen(SCREENZ);

    GsSetAmbient(ONE / 2, ONE / 2, ONE / 2);

	return ERR_NOERROR;
}

DWORD AddObject(u_long *object, WORD *ObjectID, struct POS *position, struct ROT *rotation,
	struct SCALE *scale)
{
    DWORD       loop    = 0;
    DWORD       objects = 0;
    Vertices    *Vertex;
    POLY_GT3   	*Primative[2];
    TMD_PRIM    Tmd;
	long		color;

    if (NULL == object || NULL == ObjectID || NULL == position || NULL == rotation || NULL == scale)
    {
#ifdef DEBUG
		printf("Bad Parameters to AddObject\n");
#endif
    	*ObjectID 	= 0xffff;
    	return ERR_BADPARAMS;
    }

    for (loop = 0; loop < MAXOBJECTS; loop++)
    {
    	if (FALSE == Objects[loop].Used)
    	{
    		break;
    	}
    }

    if (loop >= MAXOBJECTS)
    {
#ifdef DEBUG
        printf("No object available in AddObject\n");
#endif
    	return ERR_NONEAVAILABLE;
    }

    Objects[loop].Object[0].Primatives = OpenTMD(object, 0);

    if (Objects[loop].Object[0].Primatives > MAXPOLYGONS)
    {
#ifdef DEBUG
        printf("Too many polygons in AddObject\n");
#endif

        return ERR_INITERROR;
    }

    Vertex          = Objects[loop].Object[0].Vertex;
    Primative[0]    = Objects[loop].Object[0].Primative[0];
    Primative[1]    = Objects[loop].Object[0].Primative[1];

    for (objects = 0; objects < Objects[loop].Object[0].Primatives && ReadTMD(&Tmd) != 0; objects++)
    {
        SetPolyGT3(Primative[0]);

        copyVector(&Vertex->v0, &Tmd.x0);
        copyVector(&Vertex->v1, &Tmd.x1);
        copyVector(&Vertex->v2, &Tmd.x2);

		color = (Tmd.n0.vx + Tmd.n0.vy) * 128 / ONE / 2 + 128;
		setRGB0(Primative[0], color, color, color);
		color = (Tmd.n1.vx + Tmd.n1.vy) * 128 / ONE / 2 + 128;
		setRGB1(Primative[0], color, color, color);
		color = (Tmd.n2.vx + Tmd.n2.vy) * 128 / ONE / 2 + 128;
		setRGB2(Primative[0], color, color, color);

        setUV3(Primative[0], Tmd.u0, Tmd.v0, Tmd.u1, Tmd.v1, Tmd.u2, Tmd.v2);
        Primative[0]->tpage		= Tmd.tpage;
        Primative[0]->clut   	= Tmd.clut;

        memcpy(Primative[1], Primative[0], sizeof(POLY_GT3));

        Vertex++;
        Primative[0]++;
        Primative[1]++;
    }

    Objects[loop].Position.vx       		= position->x;
    Objects[loop].Position.vy       		= position->y;
    Objects[loop].Position.vz       		= position->z + ZSTART;
    Objects[loop].Rotation.vx       		= rotation->x * ONE / 360;
    Objects[loop].Rotation.vy       		= rotation->y * ONE / 360;
    Objects[loop].Rotation.vz       		= rotation->z * ONE / 360;
    Objects[loop].Scale.vx					= scale->x * ONE;
    Objects[loop].Scale.vy          		= scale->y * ONE;
    Objects[loop].Scale.vz          		= scale->z * ONE;
	Objects[loop].CurrentFrame				= 0;
    Objects[loop].Used 						= TRUE;
    Objects[loop].Object[0].Primatives		= objects;

	*ObjectID = loop;

	return ERR_NOERROR;
}

DWORD AddAnimation(u_long *object, WORD ObjectID, long frame)
{
    DWORD       loop    = 0;
    DWORD       objects = 0;
    Vertices    *Vertex;
    POLY_GT3   	*Primative[2];
    TMD_PRIM    Tmd;
	long		color;

    if (NULL == object)
    {
#ifdef DEBUG
		printf("Bad Parameters to AddAnimation\n");
#endif

    	return ERR_BADPARAMS;
    }

    Objects[ObjectID].Object[frame].Primatives = OpenTMD(object, 0);

    if (Objects[ObjectID].Object[frame].Primatives > MAXPOLYGONS)
    {
#ifdef DEBUG
        printf("Too many polygons in AddFrame\n");
#endif

        return ERR_INITERROR;
    }

    Vertex          = Objects[ObjectID].Object[frame].Vertex;
    Primative[0]    = Objects[ObjectID].Object[frame].Primative[0];
    Primative[1]    = Objects[ObjectID].Object[frame].Primative[1];

    for (objects = 0; objects < Objects[ObjectID].Object[frame].Primatives && ReadTMD(&Tmd) != 0;
        objects++)
    {
        SetPolyGT3(Primative[0]);

        copyVector(&Vertex->v0, &Tmd.x0);
        copyVector(&Vertex->v1, &Tmd.x1);
        copyVector(&Vertex->v2, &Tmd.x2);

		color = (Tmd.n0.vx + Tmd.n0.vy) * 128 / ONE / 2 + 128;
		setRGB0(Primative[0], color, color, color);
		color = (Tmd.n1.vx + Tmd.n1.vy) * 128 / ONE / 2 + 128;
		setRGB1(Primative[0], color, color, color);
		color = (Tmd.n2.vx + Tmd.n2.vy) * 128 / ONE / 2 + 128;
		setRGB2(Primative[0], color, color, color);

        setUV3(Primative[0], Tmd.u0, Tmd.v0, Tmd.u1, Tmd.v1, Tmd.u2, Tmd.v2);
        Primative[0]->tpage		= Tmd.tpage;
        Primative[0]->clut   	= Tmd.clut;

        memcpy(Primative[1], Primative[0], sizeof(POLY_GT3));

        Vertex++;
        Primative[0]++;
        Primative[1]++;
    }

	return ERR_NOERROR;
}

DWORD RemoveObject(DWORD ObjectID)
{
	if (ObjectID > MAXOBJECTS || FALSE == Objects[ObjectID].Used)
	{
#ifdef DEBUG
		printf("Bad Parameters to RemoveObject\n");
#endif

		return ERR_BADPARAMS;
	}

	Objects[ObjectID].Used = FALSE;

	return ERR_NOERROR;
}

DWORD MoveObject(WORD ObjectID, struct POS *position)
{
	if (ObjectID > MAXOBJECTS || FALSE == Objects[ObjectID].Used)
	{
#ifdef DEBUG
		printf("Bad Parameters to MoveObject\n");
#endif
		return ERR_BADPARAMS;
	}

    Objects[ObjectID].Position.vx	= position->x;
    Objects[ObjectID].Position.vy 	= position->y;
    Objects[ObjectID].Position.vz 	= position->z + ZSTART;

    return ERR_NOERROR;
}

DWORD RotateObject(WORD ObjectID, struct ROT *rotation)
{
{
	if (ObjectID > MAXOBJECTS || FALSE == Objects[ObjectID].Used)
	{
#ifdef DEBUG
		printf("Bad Parameters to RotateObject\n");
#endif
		return ERR_BADPARAMS;
	}

    Objects[ObjectID].Rotation.vx	= rotation->x * ONE / 360;
    Objects[ObjectID].Rotation.vy 	= rotation->y * ONE / 360;
    Objects[ObjectID].Rotation.vz 	= rotation->z * ONE / 360;

    return ERR_NOERROR;
}

}

DWORD SetObjectFrame(WORD ObjectID, long frame)
{
	if (ObjectID > MAXOBJECTS || FALSE == Objects[ObjectID].Used)
	{
#ifdef DEBUG
		printf("Bad Parameters to SetObjectFrame\n");
#endif
		return FALSE;
	}

	Objects[ObjectID].CurrentFrame = frame;

	return ERR_NOERROR;
}

BOOL CheckIntersection(WORD ObjectID, short x, short y, short range, short *distance, short *xdir,
	short *ydir)
{
	short x1;
	short y1;
	short x2;
	short y2;
	short xtotal;
	short ytotal;
	short xcenter;
	short ycenter;

	if (ObjectID > MAXOBJECTS || FALSE == Objects[ObjectID].Used || NULL == distance || NULL == xdir
		|| NULL == ydir)
	{
#ifdef DEBUG
		printf("Bad Parameters to Check Intersection\n");
#endif
		return FALSE;
	}

	x1	= Objects[ObjectID].ScreenRect.x;
	y1	= Objects[ObjectID].ScreenRect.y;
	x2	= Objects[ObjectID].ScreenRect.w;
	y2	= Objects[ObjectID].ScreenRect.h;

	if (x + range >= x1 && x - range <= x2 && y + range >= y1 && y - range <= y2)
	{
		xcenter = (x1 + x2) / 2;
		xtotal 	= (x - xcenter) ^ 2;
		ycenter = (y1 + y2) / 2;
		ytotal 	= (y - ycenter) ^ 2;

		*distance = (short)sqrt((double)(xtotal + ytotal));
		*xdir	  = x < xcenter ? -1 : 1;
		*ydir	  = y > ycenter ? -1 : 1;

		return TRUE;
	}

	return FALSE;
}

BOOL CheckFloor(WORD ObjectID, short y)
{
	long y1;
	long y2;

	if (ObjectID > MAXOBJECTS || FALSE == Objects[ObjectID].Used)
	{
#ifdef DEBUG
		printf("Bad Parameters to Check Floor\n");
#endif
		return FALSE;
	}

	y1	= Objects[ObjectID].ScreenRect.y;
	y2	= Objects[ObjectID].ScreenRect.h;

	if (y >= y1 && y <= y2)
	{
		return TRUE;
	}

	return FALSE;
}

BOOL CheckScreen(WORD ObjectID)
{
	long x1;
	long y1;
	long x2;
	long y2;

	if (ObjectID > MAXOBJECTS || FALSE == Objects[ObjectID].Used)
	{
#ifdef DEBUG
		printf("Bad Parameters to Check Screen\n");
#endif
		return FALSE;
	}

	x1	= Objects[ObjectID].ScreenRect.x;
	y1	= Objects[ObjectID].ScreenRect.y;
	x2	= Objects[ObjectID].ScreenRect.w;
	y2	= Objects[ObjectID].ScreenRect.h;

	if ((x1 < 0 && x2 < 0) || (x1 > SCREENWIDTH && x2 > SCREENWIDTH) || (y1 < 0 && y2 < 0) ||
		(y1 > SCREENHEIGHT && y2 > SCREENHEIGHT))
	{
		return TRUE;
	}

	return FALSE;
}

DWORD Render(long ActiveBuffer)
{
    Vertices    *Vertex;
    POLY_GT3   	*Primative;
    MATRIX      Matrix;
    DWORD       loop;
    DWORD       objects;
    long        temp;
    long        OtZ;
    long        interpolation;
    long        flag;
    long        objx;
    long        objy;
    long		frame;
    short		Obj;
    short		ObjID[MAXOBJECTS];
    long		zpos[MAXOBJECTS];
    long		tmp;
    BOOL		Changed = TRUE;

    for (loop = 0; loop < MAXOBJECTS; loop++)
    {
		if (FALSE == Objects[loop].Used)
		{
			ObjID[loop] = -1;
		}

		else
		{
			ObjID[loop] = loop;
		}

		zpos[loop]	= Objects[loop].Position.vz;
	}

    while (TRUE == Changed)
    {
    	Changed = FALSE;

	    for (loop = 0; loop < (MAXOBJECTS - 1); loop++)
    	{
			if (-1 == ObjID[loop] && ObjID[loop + 1] >= 0)
			{
				ObjID[loop] 	= ObjID[loop + 1];
				ObjID[loop + 1] = -1;
				zpos[loop]		= zpos[loop + 1];

				Changed = TRUE;

				continue;
			}

			if (ObjID[loop] != -1 && ObjID[loop + 1] != -1)
			{
				if (zpos[loop] > zpos[loop + 1])
				{
					tmp 			= ObjID[loop];
					ObjID[loop]		= ObjID[loop + 1];
					ObjID[loop + 1] = tmp;
					tmp 			= zpos[loop];
					zpos[loop]  	= zpos[loop + 1];
					zpos[loop + 1]  = tmp;

					Changed = TRUE;
				}
			}
		}
    }

    for (loop = 0; loop < MAXOBJECTS; loop++)
    {
		Obj	= ObjID[loop];

		if (Obj < 0)
		{
			break;
		}

	    frame		= Objects[Obj].CurrentFrame;

        RotMatrix(&Objects[Obj].Rotation, &Matrix);
        TransMatrix(&Matrix, &Objects[Obj].Position);
        ScaleMatrix(&Matrix, &Objects[Obj].Scale);
        SetRotMatrix(&Matrix);
        SetTransMatrix(&Matrix);

        Vertex      = Objects[Obj].Object[frame].Vertex;
        Primative   = Objects[Obj].Object[frame].Primative[ActiveBuffer];

		Objects[Obj].ScreenRect.x = 1000;
		Objects[Obj].ScreenRect.y = 1000;
		Objects[Obj].ScreenRect.w = -1;
		Objects[Obj].ScreenRect.h = -1;

        for (objects = 0; objects < Objects[Obj].Object[frame].Primatives; objects++,
            Vertex++, Primative++)
        {
            temp    = RotAverageNclip3(&Vertex->v0, &Vertex->v1, &Vertex->v2,
                (long *)&Objects[Obj].Object[frame].Primative[ActiveBuffer][objects].x0,
                (long *)&Objects[Obj].Object[frame].Primative[ActiveBuffer][objects].x1,
                (long *)&Objects[Obj].Object[frame].Primative[ActiveBuffer][objects].x2,
                &interpolation, &OtZ, &flag);

			OtZ /=  128;

			if (temp > 0 && OtZ > 0 && OtZ < OT_LENGTH)
			{
				GsSortPoly(&Objects[Obj].Object[frame].Primative[ActiveBuffer][objects],
					&OT[ActiveBuffer], OtZ);

				objx = Objects[Obj].Object[frame].Primative[ActiveBuffer][objects].x0;
				objy = Objects[Obj].Object[frame].Primative[ActiveBuffer][objects].y0;

				if (objx < Objects[Obj].ScreenRect.x)
				{
                        Objects[Obj].ScreenRect.x 	= objx;
				}

				if (objx > Objects[Obj].ScreenRect.w)
				{
					Objects[Obj].ScreenRect.w 		= objx;
				}

				if (objy < Objects[Obj].ScreenRect.y)
				{
					Objects[Obj].ScreenRect.y 		= objy;
				}

				if (objy > Objects[Obj].ScreenRect.h)
				{
					Objects[Obj].ScreenRect.h 		= objy;
				}
            }
        }

    }

	return ERR_NOERROR;
}

DWORD LoadTIM(u_long *TIM)
{
	TIM_IMAGE       image;

	if (NULL == TIM)
	{
#ifdef DEBUG
		printf("Bad Parameters to LoadTIM.\n");
#endif

		return ERR_BADPARAMS;
	}

	if (OpenTIM(TIM) != 0)
	{
#ifdef DEBUG
		printf("Unable to open TIM.\n");
#endif

		return ERR_FILEERROR;
	}

	while (ReadTIM(&image) != 0)
	{
		if (image.caddr != 0)
		{
			LoadImage(image.crect, image.caddr);
		}

		if (image.paddr != 0)
		{
			LoadImage(image.prect, image.paddr);
		}
	}

	return ERR_NOERROR;
}

