// **************************************************************************
// *				 Sporting Clays for the Playstation						*
// *					   (C) 1999 Charles Doty                           	*
// *	 		   Input Source file contains input code					*
// **************************************************************************

#include "input.h"

// System timing correct defines
#define NTSC_COEFFICIENT	0.198166
#define PAL_COEFFICIENT		0.196358
#define KONAMI_OFFSETX		140
#define NAMCO_OFFSETXZ1		80
#define NAMCO_OFFSETXZ2		20
#define NAMCO_OFFSETY		25

#define MOUSESCALEX			3
#define MOUSESCALEY         4
#define MOUSESPEED			2

WORD ASCII[3][2] =
{
	{0x824f, 0x30},	{0x8260, 0x41},	{0x8281, 0x61},
};

WORD SJIS[] =
{
	0x8140,	0x8149,	0x8168,	0x8194,	0x8190,	0x8193,	0x8195,	0x8166,	0x8169,	0x816a,	0x8196,	0x817b,
	0x8143,	0x817c,	0x8144,	0x815e,	0x8146,	0x8147,	0x8171,	0x8181,	0x8172,	0x8148,	0x8197,	0x816d,
	0x818f,	0x816e,	0x814f,	0x8151,	0x8165,	0x816f,	0x8162,	0x8170,	0x8150,
};

ControllerData      Controller[2];      // Buffer for both controllers
Gun                 Guns[2];            // Gun shot buffer
BOOL                Fired 		= TRUE; // Should a shot be checked for?
BOOL                MouseUsed 	= FALSE;// Is mouse being used?
BOOL                GunUsed 	= FALSE;// Is gun being used?
BOOL                NamcoUsed 	= FALSE;// Is namco gun being used?
BOOL                PadUsed 	= FALSE;// Is pad being used?
DWORD               PadXValue = 1;      // X Value of Pad movement
DWORD               PadYValue = 1;      // Y Value of Pad movement
RECT				MouseLimits;		// Limits of the mouse movement
long				MouseX;				// Current mouse x position
long				MouseY;				// Current mouse x position

DWORD InputInitialize()
{
	bzero((BYTE *)&Controller[0], sizeof(Controller[0]));
	bzero((BYTE *)&Controller[1], sizeof(Controller[1]));
	bzero((BYTE *)&Guns[0], sizeof(Guns[0]));
	bzero((BYTE *)&Guns[1], sizeof(Guns[1]));

	InitPAD((char *)&Controller[0], sizeof(Controller[0]), (char *)&Controller[1],
		sizeof(Controller[1]));

	StartPAD();

	PadUsed	= TRUE;

    MouseLimits.x	= 0;
    MouseLimits.y	= 0;
    MouseLimits.w	= SCREENWIDTH - 1;
    MouseLimits.h	= SCREENHEIGHT - 1;
	MouseX			= SCREENWIDTH / 2;
	MouseY			= SCREENHEIGHT / 2;

	return ERR_NOERROR;
}

DWORD ReadInput(InputStruct *Input, DWORD gun)
{
	Input->Buttons	= BUT_NONE;
	Input->Type		= CON_NONE;

	if (Controller[gun].Status != 0xFF)
	{
		switch ((Controller[gun].Format & 0xF0) >> 4)
		{
			case MOUSE:
				if (FALSE == MouseUsed)
				{
					if (FALSE == PadUsed)
					{
						CloseCurrentDevice();

						StartPAD();
            		}

					else
					{
						PadUsed = FALSE;
					}

					MouseUsed	= TRUE;
				}

				else
				{
					return ReadMouse(Input, gun);
				}

				break;

			case KONAMI_GUN:
				if (FALSE == GunUsed)
				{
					CloseCurrentDevice();

					InitGUN((char *)&Controller[0], sizeof(Controller[0]), (char *)&Controller[1],
						sizeof(Controller[1]), (char *)&Guns[0], (char *)&Guns[1], MAXBUFFERSHOTS);

					StartGUN();

					GunUsed = TRUE;
				}

				else
				{
					return ReadKonamiGun(Input, gun);
				}

				break;

			case PAD:
				if (FALSE == PadUsed)
				{
					CloseCurrentDevice();

					StartPAD();

					PadUsed	= TRUE;
				}

				else
				{
					return ReadPad(Input, gun);
				}

				break;

			case ANALOG:
				if (FALSE == PadUsed)
				{
					CloseCurrentDevice();

					StartPAD();

					PadUsed	= TRUE;
				}

				else
				{
					return ReadAnalog(Input, gun);
				}

				break;

			case NAMCO_GUN:
				if (FALSE == NamcoUsed)
				{
					CloseCurrentDevice();

					StartPAD();

					NamcoUsed = TRUE;
				}

				else
				{
					return ReadNamcoGun(Input, gun);
				}

				break;

			case MULTITAP:
				break;
		}
	}

	else
	{
		ResetCallback();
		CloseCurrentDevice();

		PadUsed = TRUE;

		StartPAD();
	}

	return ERR_NOERROR;
}

DWORD SetPadMoveValues(DWORD x, DWORD y)
{
	PadXValue = x;
	PadYValue = y;

	return ERR_NOERROR;
}

DWORD ReadPad(InputStruct *Input, DWORD controller)
{
	if (0 == (Controller[controller].InputType.pad & PAD_X))
	{
		if (TRUE == Fired)
		{
			Input->Type		= CON_PAD;
			Input->Buttons	|= BUT_TRIGGER;

			Fired = FALSE;
		}
	}

	else
	{
		Fired = TRUE;
	}

	if (0 == (Controller[controller].InputType.pad & PAD_START))
	{
		Input->Type		= CON_PAD;
		Input->Buttons	|= BUT_START;
	}

	if (0 == (Controller[controller].InputType.pad & PAD_UP))
	{
		Input->Type		= CON_PAD;
		MouseY			-= PadYValue * MOUSESCALEY;
	}

	if (0 == (Controller[controller].InputType.pad & PAD_DOWN))
	{
		Input->Type		= CON_PAD;
		MouseY			+= PadYValue * MOUSESCALEY;
	}

	if (0 == (Controller[controller].InputType.pad & PAD_LEFT))
	{
		Input->Type		= CON_PAD;
		MouseX			-= PadXValue * MOUSESCALEX;
	}

	if (0 == (Controller[controller].InputType.pad & PAD_RIGHT))
	{
		Input->Type		= CON_PAD;
		MouseX			+= PadXValue * MOUSESCALEX;
	}

	if (MouseX < MouseLimits.x)
	{
		MouseX = MouseLimits.x;
	}

	else if (MouseX > MouseLimits.w)
	{
		MouseX = MouseLimits.w;
	}

	if (MouseY < MouseLimits.y)
	{
		MouseY = MouseLimits.y;
	}

	else if (MouseY > MouseLimits.h)
	{
		MouseY = MouseLimits.h;
	}

	Input->X	= MouseX / MOUSESCALEX;
	Input->Y	= MouseY / MOUSESCALEY;

	return ERR_NOERROR;
}

DWORD ReadAnalog(InputStruct *Input, DWORD controller)
{
	long	x = Controller[controller].InputType.analog.LeftX;
	long	y = Controller[controller].InputType.analog.LeftY;

	if (0 == (Controller[controller].InputType.pad & PAD_X))
	{
		if (TRUE == Fired)
		{
			Input->Type		= CON_PAD;
			Input->Buttons	|= BUT_TRIGGER;

			Fired = FALSE;
		}
	}

	else
	{
		Fired = TRUE;
	}

	if (0 == (Controller[controller].InputType.pad & PAD_START))
	{
		Input->Type		= CON_PAD;
		Input->Buttons	|= BUT_START;
	}

	if (x < 128 - 32 || x > 128 + 32)
	{
		if (x < 128)
		{
			x += 32;
		}

		if (x > 128)
		{
			x -= 32;
		}

		Input->Type		= CON_PAD;
		MouseX			-= ((128 - x) / 16) * MOUSESCALEY;
	}

	if (y < 128 - 32 || y > 128 + 32)
	{
		if (y < 128)
		{
			y += 32;
		}

		if (y > 128)
		{
			y -= 32;
		}

		Input->Type		= CON_PAD;
		MouseY			-= ((128 - y) / 16) * MOUSESCALEY;
	}

	if (MouseX < MouseLimits.x)
	{
		MouseX = MouseLimits.x;
	}

	else if (MouseX > MouseLimits.w)
	{
		MouseX = MouseLimits.w;
	}

	if (MouseY < MouseLimits.y)
	{
		MouseY = MouseLimits.y;
	}

	else if (MouseY > MouseLimits.h)
	{
		MouseY = MouseLimits.h;
	}

	Input->X	= MouseX / MOUSESCALEX;
	Input->Y	= MouseY / MOUSESCALEY;

	return ERR_NOERROR;
}

DWORD ReadMouse(InputStruct *Input, DWORD controller)
{
	if (Controller[controller].InputType.mouse.X != 0 ||
		Controller[controller].InputType.mouse.Y != 0)
	{
		MouseX		+= Controller[controller].InputType.mouse.X * MOUSESPEED;
		MouseY		+= Controller[controller].InputType.mouse.Y * MOUSESPEED;

		if (MouseX < MouseLimits.x)
		{
			MouseX = MouseLimits.x;
		}

		else if (MouseX > MouseLimits.w)
		{
			MouseX = MouseLimits.w;
		}

		if (MouseY < MouseLimits.y)
		{
			MouseY = MouseLimits.y;
		}

		else if (MouseY > MouseLimits.h)
		{
			MouseY = MouseLimits.h;
		}

		Input->X	= MouseX / MOUSESCALEX;
		Input->Y	= MouseY / MOUSESCALEY;
		Input->Type	= CON_MOUSE;
	}

	if (0 == (Controller[controller].InputType.pad & MOUSE_LEFTBUTTON))
	{
		if (TRUE == Fired)
		{
			Input->Buttons	|= BUT_LEFT;
			Input->Type		= CON_MOUSE;

			Fired = FALSE;
		}
	}

	else
	{
		Fired = 1;
	}

	if (0 == (Controller[controller].InputType.pad & MOUSE_RIGHTBUTTON))
	{
		Input->Buttons	|= BUT_RIGHT;
		Input->Type		= CON_MOUSE;
	}

	return ERR_NOERROR;
}

DWORD ReadKonamiGun(InputStruct *Input, DWORD controller)
{
	SelectGUN(controller, TRUE);

	if (0 == (Controller[controller].InputType.pad & KONAMI_TRIGGER))
	{
		if (TRUE == Fired)
		{
			Input->X		= Guns[0].Shots[0].XTiming - KONAMI_OFFSETX;

			if (TRUE == IsNTSC())
			{
				Input->X	= (long)((float)Input->X * NTSC_COEFFICIENT);
			}

			else
			{
				Input->X	= (long)((float)Input->X * PAL_COEFFICIENT);
			}

			Input->Y		= Guns[0].Shots[0].YTiming;
			Input->Buttons	|= BUT_TRIGGER;
			Input->Type		= CON_GUN;

			Fired			= FALSE;
		}
	}

	else
	{
		Fired			= TRUE;
	}

	if (0 == (Controller[controller].InputType.pad & KONAMI_START))
	{
		Input->Buttons |= BUT_START;
		Input->Type		= CON_GUN;
	}

	return ERR_NOERROR;
}

DWORD ReadNamcoGun(InputStruct *Input, DWORD controller)
{
	if (0 == (Controller[controller].InputType.pad & GUNCON_TRIGGER))
	{
		if (TRUE == Fired)
		{
			BYTE Section	= Controller[controller].InputType.gun.Section;
			Input->X		= (BYTE)Controller[controller].InputType.gun.X;
			Input->Y		= (BYTE)Controller[controller].InputType.gun.Y - NAMCO_OFFSETY;

			if (0 == Section)
			{
				Input->X	= Input->X;
				Input->X	-= NAMCO_OFFSETXZ1;
			}

			else
			{
				Input->X	+= 160 - NAMCO_OFFSETXZ2;
			}

			Input->Buttons	|= BUT_TRIGGER;
			Input->Type		= CON_GUN;

			Fired			= FALSE;
		}
	}

	else
	{
		Fired			= TRUE;
	}

	if (0 == (Controller[controller].InputType.pad & GUNCON_START))
	{
		Input->Buttons	|= BUT_START;
		Input->Type		= CON_GUN;
	}

	return ERR_NOERROR;
}

DWORD CloseCurrentDevice()
{
	if (TRUE == PadUsed)
	{
		StopPAD();

		PadUsed		= FALSE;
	}

	if (TRUE == MouseUsed)
	{
		StopPAD();

		MouseUsed	= FALSE;
	}

	if (TRUE == GunUsed)
	{
		StopGUN();

		GunUsed 	= FALSE;
	}

	if (TRUE == NamcoUsed)
	{
		StopPAD();

		NamcoUsed 	= FALSE;
	}

	return ERR_NOERROR;
}

DWORD SetMouseLimits(long x1, long y1, long x2, long y2)
{
    MouseLimits.x	= x1 * MOUSESCALEX;
    MouseLimits.y	= y1 * MOUSESCALEY;
    MouseLimits.w	= x2 * MOUSESCALEX;
    MouseLimits.h	= y2 * MOUSESCALEY;

	return ERR_NOERROR;
}

DWORD SetInputMousePosition(long x, long y)
{
	MouseX			= x * MOUSESCALEX;
	MouseY			= y * MOUSESCALEY;

	return ERR_NOERROR;
}

DWORD Convert2SJIS(BYTE *string, BYTE *dest)
{
	int		loop;
	int 	sjis_code;
	int 	ascii_code;
	BYTE 	stmp;
	BYTE 	stmp2;
	BYTE 	*dest2;

	dest2 = dest;

	for (loop = 0; loop < 32; loop++)
	{
		*dest2++ = 0x81;
		*dest2++ = 0x40;
	}

	while (*string != 0)
	{
		stmp2 		= 0;
		ascii_code 	= *string++;

		if ((ascii_code >= 0x20) && (ascii_code <= 0x2f))
		{
			stmp2 = 1;
		}

		else if ((ascii_code >= 0x30) && (ascii_code <= 0x39))
		{
			stmp = 0;
		}

		else if ((ascii_code >= 0x3a) && (ascii_code <= 0x40))
		{
			stmp2 = 11;
		}

		else if ((ascii_code >= 0x41) && (ascii_code <= 0x5a))
		{
			stmp = 1;
		}

		else if ((ascii_code >= 0x5b) && (ascii_code <= 0x60))
		{
			stmp2 = 37;
		}

		else if ((ascii_code >= 0x61) && (ascii_code <= 0x7a))
		{
			stmp = 2;
		}

		else if ((ascii_code >= 0x7b) && (ascii_code <= 0x7e))
		{
			stmp2 = 63;
		}

		else
		{
#ifdef DEBUG
			printf("Bad ASCII code 0x%x to Convert2SJIS.\n", ascii_code);
#endif

			return ERR_BADPARAMS;
		}

		if (stmp2 != 0)
		{
			sjis_code = SJIS[ascii_code - 0x20 - (stmp2 - 1)];
		}

		else
		{
			sjis_code = ASCII[stmp][0] + ascii_code - ASCII[stmp][1];
     	}

		*dest++ = (sjis_code & 0xff00) >> 8;
		*dest++ = (sjis_code & 0xff);
	}

	return ERR_NOERROR;
}
