// **********************************************************************************************
// *								  3D Engine													*
// **********************************************************************************************

#ifndef __SCENE3D_H
#define __SCENE3D_H

#include "Camera.h"
#include "Frustum.h"
#include "Meshes.h"
#include "MLFormat.h"
#include "Types.h"

#define	SHOWFRAMERATE

#define	TYPE_MESH		0x00000000					// Mesh	type
#define	TYPE_CHARACTER	0x00000001					// Character type
#define	TYPE_LIGHT		0x00000002					// Light

#define	SELECTSIZE		512							// Selection buffer size

struct tagSort
{
	DWORD				z;							// Z position
	DWORD				Type;						// Mesh type
	DWORD				Index;						// Mesh index
};

struct tagPath
{
	TVertex				Current;					// Current position	
	TVertex				Delta;						// Delta position 
	DWORD				Steps;						// Steps left
	DWORD				LandscapeID;				// Landscape ID
	BOOL				Enabled;					// Is the animation enabled
	DWORD				AnimationFlag;				// Animation flag (ANIM_PLAYONCE...)
	char				UpperAnimation[MAX_PATH];	// Ending upper animation
	char				LowerAnimation[MAX_PATH];	// Ending lower animation
};

struct tagTexture
{
	UINT				TextureID;					// OpenGL textures numbers
	char				Name[MAX_PATH];				// Name
};

class TScene3D
{
  protected:
	TCamera				*Cameras[MAXCAMERAS];		// Camera
	DWORD				NumCameras;					// Number of cameras
	TFrustum			*Frustum;					// Frustum
	TBaseMesh3D			*Meshes[MAXMESHES];			// Meshes
	DWORD				NumMeshes;					// Number of meshes
	TCharacterMesh3D	*Characters[MAXCHARACTERS];	// Characters
	DWORD				NumCharacters;				// Number of characters
	TSectionData3D		*SectionData[MAXSECTIONS];	// Section data
	DWORD				NumSections;				// Number of sections
	tagPath				Paths[MAXCHARACTERS];		// Character paths
	TLightMesh3D		*Lights[MAXLIGHTS];			// Lights
	DWORD				NumLights;					// Number of lights
	DWORD				NumTextures;				// Number of textures
	tagSort				SortOrder[MAXMESHES + MAXCHARACTERS];// Mesh sort order
	tagTexture			Textures[MAXTEXTURES];		// Texture info
	DWORD				CurrentSort;				// Current sort position
	DWORD				CurrentCamera;				// Current camera
	BOOL				CameraMoved;				// Has the camera moved?
	float				FrameInterval;				// This stores our frame interval between frames

  public:
	TScene3D();
	  // Constructor

	~TScene3D();
	  // Destructor

	DWORD Initialize();
	  // Initialization routine. Sets up the scene.

	DWORD Close();
	  // Close routine. Close the scene.

	DWORD Animate(float DeltaTime);
	  // Animate the scene

	DWORD Render();
	  // Render the scene.
	
	DWORD RenderGrid(DWORD Mesh);
	  // Render a grid (based on the landscape)

	DWORD Reset();
	  // Reset the scene.
	
	DWORD Load(char *Filename);
	  // Load a mesh list

	DWORD LoadCharacter(char *Filename, char *Name, DWORD *ObjectID = NULL);
	  // Load a character mesh

	DWORD ReadLights(BYTE **Buffer);
	  // Read lights

	DWORD ReadCameras(BYTE **Buffer);
	  // Read cameras

	DWORD ReadMeshSection(BYTE **Buffer, DWORD *SectionID);
	  // Read a mesh section

	DWORD ReadCharacterSection(BYTE **Buffer, DWORD *SectionID);
	  // Read a character section

	DWORD AddToSortOrder(DWORD z, DWORD Type, DWORD Index);
	  // Add to sort order	
	
	DWORD SortObjects();
	  // Sort objects 

	DWORD RenderSortOrder();
	  // Render for the sort order
	
	UINT GetTexture(DWORD TextureID);
	  // Get OpenGL texture id
	
	DWORD FindSceneItem(DWORD Type, char *Name);
	  // Find a scene based on name and type

	DWORD AddCamera(TVertex *Position, TVertex *LookAt, TVertex *UpVector);
	  // Add a camera 

	DWORD SetCurrentCamera(DWORD Camera);
	  // Set current camera

	DWORD RotateCurrentCamera(float Angle, float x, float y, float z);
	  // Rotate current camera

	DWORD MoveCurrentCamera(float Distance, BOOL Up = FALSE);
	  // Move current camera

	DWORD SetCurrentCamera(char *Name);
	  // Set current camera by name

	DWORD GetCameraPosition(TVertex *Position);
	  // Get camera position

	DWORD SetCameraPosition(TVertex *Position);
	  // Set camera position

	DWORD GetCameraLookAt(TVertex *LookAt);
	  // Get camera look at

	DWORD SetCameraLookAt(TVertex *LookAt);
	  // Set camera look at

	DWORD GetCameraUpVector(TVertex *UpVector);
	  // Get camera up vector

	DWORD SetCameraUpVector(TVertex *UpVector);
	  // Set camera up vector

	DWORD SetCameraEditMode(BOOL EditMode);
	  // Set camera edit mode

	TFrustum *GetFrustum();
	  // Get frustum

	DWORD ResetFrustum();
	  // Reset frustum

	DWORD EnableAnimation(DWORD Type, DWORD Mesh);
	  // Enable animation

	DWORD DisableAnimation(DWORD Type, DWORD Mesh);
	  // Disable animation

	DWORD GetAnimationLength(DWORD Type, DWORD Mesh);
	  // Get animation length
	    
	DWORD SetAnimation(DWORD Mesh, DWORD Part, char *Name);
	  // Set animation based on name

	DWORD SetAnimationFrames(DWORD Type, DWORD Mesh, DWORD NewStartFrame, DWORD NewEndFrame);
	  // Set start and end frames for an animation

	DWORD SetAnimationType(DWORD Type, DWORD Mesh, DWORD NewType);
	  // Set start and end frames for an animation

	DWORD SetAnimationSpeed(DWORD Type, DWORD Mesh, float NewRate);
	  // Set animation speed

	DWORD GetPosition(DWORD Type, char *Name, TVertex *Position);
	  // Get helper position

	DWORD GetPosition(DWORD Type, DWORD Mesh, TVertex *Position);
	  // Get position

	DWORD GetPosition(DWORD Type, DWORD Mesh, char *Name, TVertex *Position);
	  // Get position (of a sub object).

	DWORD SetPosition(DWORD Type, DWORD Mesh, TVertex *Position);
	  // Set position
	
	DWORD GetRotation(DWORD Type, DWORD Mesh, TVertex *Rotation);
	  // Get position

	DWORD SetRotation(DWORD Type, DWORD Mesh, TVertex *Rotation);
	  // Set position
	
	DWORD SetCharacterPath(DWORD Mesh, char *Start, char *End, char *LowerAnimation, 
		char *UpperAnimation, DWORD EndAnimationFlag, float Speed, DWORD Landscape);
	  // Set character path

	DWORD AnimateCharacter(DWORD Mesh);
	  // Start the character path animation

	DWORD StopCharacter(DWORD Mesh);
	  // Stop the character path animation

	DWORD AddObject(char *MeshName, char *Filename, TVertex *NewPosition, 
		TVertex *NewRotation = NULL, DWORD *ObjectID = NULL);
	  // Add mesh list object

	DWORD AddCharacter(char *MeshName, char *Filename, TVertex *NewPosition, 
		TVertex *NewRotation = NULL, DWORD *ObjectID = NULL);
	  // Add character object

	DWORD RemoveObject(DWORD Type, DWORD Index);
	  // Remove an object

	DWORD FindObject(DWORD *Type, DWORD *Index, DWORD *Part, TVertex *NewPosition);
	  // Add mesh list object

	DWORD SelectObject(DWORD Type, DWORD Index);
	  // Select an object
	
	DWORD DeselectObject(DWORD Type, DWORD Index);
	  // Deselect an object
	
	DWORD AddLandscape(char *MeshName, char *HeightMap, TVertex *NewPosition, DWORD Width, 
		DWORD Height, char *Texture, char *Detail, float Scale, float GroundScale,
		float DetailScale, DWORD *ObjectID = NULL);
	  // Add landscape

	DWORD AddWater(char *MeshName, TVertex *NewPosition, DWORD Width, DWORD Height, 
		UINT TextureID, float Scale, float WaterScale);
	  // Add water

	DWORD AddTree(char *MeshName, TVertex *Position, float Width, float Height, UINT TextureID);
	  // Add a tree to the landscape

	DWORD GetLandscapeHeight(DWORD Mesh, TVertex *Height);
	  // Gets landscape height

	DWORD GetWaterHeight(DWORD Mesh, TVertex *Height);
	  // Gets water height

	DWORD ReadTextures(char *Filename, BYTE **Buffer, BOOL Clamp);
	  // Read textures

	DWORD LoadTexture(char *Filename, BOOL Clamp, DWORD *TextureID = NULL);
	  // Load a texture from a TGA image

	DWORD RemoveTexture(DWORD TextureID);
	  // Remove a texture

	DWORD FindTexture(char *Filename, DWORD *TextureID);
	  // Find a texture ID for a filename

	DWORD GetNextMesh();
	  // Get next available mesh

	DWORD GetNextCharacter();
	  // Get next available character

	DWORD CreateOctree();
	  // Create an octree from the scene
#ifdef SHOWFRAMERATE
	DWORD TScene3D::CalculateFrameRate();
#endif
};

#endif