// **********************************************************************************************
// *								  3D Engine													*
// **********************************************************************************************

#include <windows.h>
#include <stdio.h>
#include <math.h>

#include "Include.h"
#include "Meshes.h"
#include "Scene3D.h"
#include "Utility\Utility.h"

TScene3D::TScene3D()
{
	Frustum	= NULL;

	for (DWORD Loop = 0; Loop < MAXCAMERAS; Loop++)
	{
		Cameras[Loop]		= NULL;
	}

	NumCameras				= 0;

	for (Loop = 0; Loop < MAXMESHES; Loop++)
	{
		Meshes[Loop]		= NULL;
	}
	
	NumMeshes				= 0;

	for (Loop = 0; Loop < MAXCHARACTERS; Loop++)
	{
		Characters[Loop]	= NULL;
	}
	
	NumCharacters			= 0;

	for (Loop = 0; Loop < MAXSECTIONS; Loop++)
	{
		SectionData[Loop]	= NULL;
	}
	
	NumSections			= 0;

	for (Loop = 0; Loop < MAXLIGHTS; Loop++)
	{
		Lights[Loop]		= NULL;
	}

	NumLights				= 0;

	for (Loop = 0; Loop < MAXCAMERAS; Loop++)
	{
		Cameras[Loop]		= NULL;
	}

	NumCameras				= 0;

	for (Loop = 0; Loop < MAXTEXTURES; Loop++)
	{
		Textures[Loop].TextureID	= -1;
	}

	NumTextures				= 0;

	for (Loop = 0; Loop < MAXCHARACTERS; Loop++)
	{
		Paths[Loop].Enabled		= FALSE;
		Paths[Loop].LandscapeID	= INVALID;
	}	

	CurrentSort		= 0;
	CurrentCamera	= 0;
	CameraMoved		= TRUE;
	FrameInterval	= 0.0f;
 }

TScene3D::~TScene3D()
{
	Close();
}

DWORD TScene3D::Initialize()
{
	Frustum	= new TFrustum;

	if (NULL == Frustum)
	{
		return	ERR_MEMORYERROR;
	}

	ResetFrustum();

	return	ERR_NOERROR;
}

DWORD TScene3D::Close()
{	
	if (Frustum != NULL)
	{
		delete	Frustum;

		Frustum		= NULL;
	}

	for (DWORD Loop = 0; Loop < MAXCAMERAS; Loop++)
	{
		delete	Cameras[Loop];

		Cameras[Loop]	= NULL;
	}

	NumCameras	= 0;

	for (Loop = 0; Loop < MAXMESHES; Loop++)
	{
		if (Meshes[Loop] != NULL)
		{
			delete	Meshes[Loop];
			Meshes[Loop]	= NULL;
		}
	}

	NumMeshes	= 0;

	for (Loop = 0; Loop < MAXCHARACTERS; Loop++)
	{
		if (Characters[Loop] != NULL)
		{
			delete	Characters[Loop];
			Characters[Loop]	= NULL;
		}
	}

	NumCharacters	= 0;

	for (Loop = 0; Loop < MAXSECTIONS; Loop++)
	{
		if (SectionData[Loop] != NULL)
		{
			delete	SectionData[Loop];
			SectionData[Loop]	= NULL;
		}
	}

	NumSections	= 0;

	for (Loop = 0; Loop < MAXLIGHTS; Loop++)
	{
		if (Lights[Loop] != NULL)
		{
			delete	Lights[Loop];
			
			Lights[Loop]		= NULL;
		}
	}

	NumLights		= 0;

	for (Loop = 0; Loop < MAXTEXTURES; Loop++)
	{
		if (Textures[Loop].TextureID != -1)
		{
			glDeleteTextures(1, &Textures[Loop].TextureID);
		}
	}

	return	ERR_NOERROR;
}

DWORD TScene3D::Animate(float DeltaTime)
{
	for (DWORD Loop = 0; Loop < NumCharacters; Loop++)
	{
		if (FALSE == Paths[Loop].Enabled)
		{
			continue;
		}

		if (Paths[Loop].Steps > 0)
		{
			TVertex	Height;

			Paths[Loop].Steps--;

			Paths[Loop].Current.x	+= Paths[Loop].Delta.x;
			Paths[Loop].Current.y	+= Paths[Loop].Delta.y;
			Paths[Loop].Current.z	+= Paths[Loop].Delta.z;

			Height.x				= Paths[Loop].Current.x;
			Height.z				= Paths[Loop].Current.z;
			GetLandscapeHeight(Paths[Loop].LandscapeID, &Height);
			Height.y				+= YOFFSET * MD3SCALE;

			Characters[Loop]->SetPosition(&Height);
		}

		else
		{
			Characters[Loop]->SetAnimation(PART_UPPER, Paths[Loop].UpperAnimation);
			Characters[Loop]->SetAnimation(PART_LOWER, Paths[Loop].LowerAnimation);
			Characters[Loop]->SetAnimationType(Paths[Loop].AnimationFlag);

			Paths[Loop].Enabled	= FALSE;
		}
	}

	if (TRUE == CameraMoved && FALSE == Cameras[CurrentCamera]->GetEditMode())
	{
		for (DWORD Loop = 0; Loop < NumLights; Loop++)
		{
			Lights[Loop]->Cull(Frustum);
		}

		for (Loop = 0; Loop < NumMeshes; Loop++)
		{
			if (Meshes[Loop] != NULL)
			{
				Meshes[Loop]->Cull(Frustum);
			}
		}

		for (Loop = 0; Loop < NumCharacters; Loop++)
		{
			if (Characters[Loop] != NULL)
			{
				Characters[Loop]->Cull(Frustum);
			}
		}
	}

	for (Loop = 0; Loop < NumLights; Loop++)
	{
		if (Lights[Loop] != NULL)
		{
			Lights[Loop]->Animate();
		}
	}

	for (Loop = 0; Loop < NumMeshes; Loop++)
	{
		if (Meshes[Loop] != NULL)
		{
			Meshes[Loop]->Animate();
		}
	}

	for (Loop = 0; Loop < NumCharacters; Loop++)
	{
		if (Characters[Loop] != NULL)
		{
			Characters[Loop]->Animate();
		}
	}

	return	ERR_NOERROR;
}

DWORD TScene3D::Render()
{
	/*
	if (TRUE == CameraMoved)
	{
		DWORD	CurrentLight	= 0;
	
		CurrentSort				= 0;
	
		Reset();

		for (DWORD Loop = 0; Loop < NumLights; Loop++)
		{
			Lights[Loop]->Render(Cameras[CurrentCamera], &CurrentLight);
		}

		for (Loop = 0; Loop < NumMeshes; Loop++)
		{
			if (Meshes[Loop] != NULL)
			{
				DWORD	z	= Meshes[Loop]->GetSortZ();

				AddToSortOrder(z, TYPE_MESH, Loop);
			}
		}

		for (Loop = 0; Loop < NumCharacters; Loop++)
		{
			if (Characters[Loop] != NULL)
			{
				DWORD	z	= Characters[Loop]->GetSortZ();
	
				AddToSortOrder(z, TYPE_CHARACTER, Loop);
			}
		}

		SortObjects();

		CameraMoved	= FALSE;
	}
	
	RenderSortOrder();
*/

	for (DWORD Loop = 0; Loop < 4; Loop++)
	{
		Meshes[Loop]->Render(Cameras[CurrentCamera]);
	}
	
	glDepthFunc(GL_LEQUAL);

	for (Loop = 4; Loop < NumMeshes; Loop++)
	{
		Meshes[Loop]->Render(Cameras[CurrentCamera]);
	}

/*
	glDepthFunc(GL_ALWAYS);

	for (DWORD Loop = 0; Loop < 24; Loop++)
	{
		Meshes[Loop]->Render(Cameras[CurrentCamera]);
	}

	glDepthFunc(GL_LESS);

	for (Loop = 24; Loop < NumMeshes; Loop++)
	{
		Meshes[Loop]->Render(Cameras[CurrentCamera]);
	}
*/
/*
	for (DWORD Loop = 8; Loop < 9; Loop++)
	{
		Meshes[Loop]->Render(Cameras[CurrentCamera]);
	}
*/

#ifdef SHOWFRAMERATE
	CalculateFrameRate();
#endif
	
	return	ERR_NOERROR;
}

DWORD TScene3D::RenderGrid(DWORD Mesh)
{
	((TGroundMesh3D *)Meshes[Mesh])->RenderGrid();

	return	ERR_NOERROR;
}

DWORD TScene3D::Reset()
{
	glLoadIdentity();
	Cameras[CurrentCamera]->Render();

	return	ERR_NOERROR;
}

DWORD TScene3D::Load(char *Filename)
{
	BYTE	*Buffer;
	BYTE	*BufferPtr;
	DWORD	MeshType;
	DWORD	TextureID;
	DWORD	ReadMeshes;

	FILE	*Handle	= fopen(Filename, "rb");

	if (NULL == Handle)
	{
		return	ERR_FILEERROR;
	}

	fseek(Handle, 0, SEEK_END);
	
	DWORD	Length	= ftell(Handle);

	rewind(Handle);

	Buffer	= new BYTE[Length];

	if (NULL == Buffer)
	{
		fclose(Handle);

		return	ERR_MEMORYERROR;
	}

	fread(Buffer, Length, 1, Handle);

	fclose(Handle);

	BufferPtr	= Buffer;

	memcpy(&ReadMeshes, BufferPtr, sizeof(DWORD));
	BufferPtr	+= sizeof(DWORD);

	for (DWORD Loop = 0; Loop < ReadMeshes; Loop++)
	{
		DWORD	MeshID	= GetNextMesh();

		if (INVALID == MeshID)
		{
			return	ERR_MEMORYERROR;
		}

		memcpy(&MeshType, BufferPtr, sizeof(DWORD));
		BufferPtr	+= sizeof(DWORD);

		switch (MeshType)
		{
			case MESH_STANDARD:
				Meshes[MeshID]	= (TBaseMesh3D *)new TStandardMesh3D;

				break;

			case MESH_ALPHA:
				Meshes[MeshID]	= (TBaseMesh3D *)new TAlphaMesh3D;

				break;

			case MESH_SKYBOX:
				Meshes[MeshID]	= (TBaseMesh3D *)new TSkyboxMesh3D;

				break;

			case MESH_TREELINE:
				Meshes[MeshID]	= (TBaseMesh3D *)new TTreelineMesh3D;

				break;

			case MESH_GROUND:
				Meshes[MeshID]	= (TBaseMesh3D *)new TGroundMesh3D;

				break;

			case MESH_WATER:
				Meshes[MeshID]	= (TBaseMesh3D *)new TWaterMesh3D;

				break;

			case MESH_BILLBOARD:
				Meshes[MeshID]	= (TBaseMesh3D *)new TBillboardMesh3D;

				break;
		
			case MESH_HELPER:
				Meshes[MeshID]	= (TBaseMesh3D *)new THelperMesh3D;

				break;

			case MESH_SHAPE:
				Meshes[MeshID]	= (TBaseMesh3D *)new TShapeMesh3D;

				break;

			case MESH_OCTREE:
				Meshes[MeshID]	= (TBaseMesh3D *)new TOctree3D;

				break;

			case MESH_OPTIMIZED:
				Meshes[MeshID]	= (TBaseMesh3D *)new TOptimized3D;

				break;

			case MESH_P3D:
				Meshes[MeshID]	= (TBaseMesh3D *)new TMeshP3D;

				break;

			case MESH_AMS:
				Meshes[MeshID]	= (TBaseMesh3D *)new TMeshAMS;

				break;
		}

		if (NULL == Meshes[MeshID])
		{
			return	ERR_MEMORYERROR;
		}

		Meshes[MeshID]->Initialize();
		FindTexture(Filename, &TextureID);

		Meshes[MeshID]->Load(&BufferPtr, TextureID);
	}

	// Read texture images
	if (MESH_SKYBOX == MeshType || MESH_TREELINE == MeshType)
	{
		ReadTextures(Filename, &BufferPtr, TRUE);
	}

	else
	{
		ReadTextures(Filename, &BufferPtr, FALSE);
	}

	// Read scene lights
	ReadLights(&BufferPtr);

	// Read cameras
	ReadCameras(&BufferPtr);

	delete [] Buffer;

	DWORD	CurrentLight	= 0;

	return	ERR_NOERROR;
}

DWORD TScene3D::LoadCharacter(char *Filename, char *Name, DWORD *ObjectID)
{
	BYTE	*Buffer;
	BYTE	*BufferPtr;
	DWORD	NumParts;

	FILE	*Handle	= fopen(Filename, "rb");

	if (NULL == Handle)
	{
		return	ERR_FILEERROR;
	}

	fseek(Handle, 0, SEEK_END);
	
	DWORD	Length	= ftell(Handle);

	rewind(Handle);

	Buffer	= new BYTE[Length];

	if (NULL == Buffer)
	{
		fclose(Handle);

		return	ERR_MEMORYERROR;
	}

	fread(Buffer, Length, 1, Handle);

	fclose(Handle);

	BufferPtr	= Buffer;

	DWORD	CharacterID	= GetNextCharacter();

	if (INVALID == CharacterID)
	{
		return	ERR_MEMORYERROR;
	}

	memcpy(&NumParts, BufferPtr, sizeof(DWORD));
	BufferPtr	+= sizeof(DWORD);

	Characters[CharacterID]	= new TCharacterMesh3D;
	Characters[CharacterID]->Initialize(Name, NumParts);

	for (DWORD Loop = 0; Loop < NumParts; Loop++)
	{
		DWORD	SectionID;
		
		ReadCharacterSection(&BufferPtr, &SectionID);

		Characters[CharacterID]->Add(Loop, SectionData[SectionID]);
	}

	// Read texture images
	ReadTextures(Filename, &BufferPtr, FALSE);

	delete [] Buffer;

	if (ObjectID != NULL)
	{
		*ObjectID	= CharacterID;
	}

	return	ERR_NOERROR;
}

DWORD TScene3D::ReadLights(BYTE **Buffer)
{
	DWORD	LightType;

	memcpy(&NumLights, *Buffer, sizeof(DWORD));
	*Buffer	+= sizeof(DWORD);

	for (DWORD Loop = 0; Loop < NumLights; Loop++)
	{
		memcpy(&LightType, *Buffer, sizeof(DWORD));
		*Buffer	+= sizeof(DWORD);

		switch (LightType)
		{
			case MESH_OMNI:
				Lights[Loop]	= (TLightMesh3D *)new TOmniMesh3D;
				Lights[Loop]->Initialize();
				Lights[Loop]->Load(Buffer, NumTextures);

				break;

			case MESH_SPOT:
				Lights[Loop]	= (TLightMesh3D *)new TSpotMesh3D;
				Lights[Loop]->Initialize();
				Lights[Loop]->Load(Buffer, NumTextures);

				break;

			case MESH_DIRECT:
				Lights[Loop]	= (TLightMesh3D *)new TDirectMesh3D;
				Lights[Loop]->Initialize();
				Lights[Loop]->Load(Buffer, NumTextures);

				break;
		}	
	}

	return	ERR_NOERROR;
}

DWORD TScene3D::ReadCameras(BYTE **Buffer)
{
	memcpy(&NumCameras, *Buffer, sizeof(DWORD));
	*Buffer	+= sizeof(DWORD);

	for (DWORD Loop = 0; Loop < NumCameras; Loop++)
	{
		Cameras[Loop]	= new TCamera;
		Cameras[Loop]->Load(Buffer);
	}

	return	ERR_NOERROR;
}

DWORD TScene3D::ReadMeshSection(BYTE **Buffer, DWORD *SectionID)
{
	return	ERR_NOERROR;
}

DWORD TScene3D::ReadCharacterSection(BYTE **Buffer, DWORD *SectionID)
{
	DWORD	TextureID;
	
	for (DWORD Loop = 0; Loop < NumSections; Loop++)
	{
		if (TRUE == SectionData[Loop]->FindName((char *)*Buffer))
		{
			if (SectionID != NULL)
			{
				*SectionID	= Loop;
			
				SectionData[Loop]->Skip(Buffer);

				return	ERR_NOERROR;
			}
		}
	}

	FindTexture((char *)*Buffer, &TextureID);

	SectionData[NumSections]	= new TSectionData3D;

	if (NULL == SectionData[NumSections])
	{
		return	ERR_MEMORYERROR;
	}

	if (SectionID != NULL)
	{
		*SectionID	= NumSections;
	}

	SectionData[NumSections]->Initialize();
	SectionData[NumSections]->Load(Buffer, TextureID);
	
	NumSections++;

	return	ERR_NOERROR;
}

DWORD TScene3D::AddToSortOrder(DWORD z, DWORD Type, DWORD Index)
{
	if (z != POS_CULLED)
	{
		SortOrder[CurrentSort].z		= z;
		SortOrder[CurrentSort].Type		= Type;
		SortOrder[CurrentSort].Index	= Index;

		CurrentSort++;
	}

	return	ERR_NOERROR;
}

DWORD TScene3D::SortObjects()
{
	BOOL	Changed	= TRUE;

	while (TRUE == Changed)
	{
		Changed	= FALSE;
	
		for (DWORD Loop = 1; Loop < CurrentSort; Loop++)
		{
			if (SortOrder[Loop - 1].z < SortOrder[Loop].z)
			{
				tagSort	Temp;

				memcpy(&Temp, &SortOrder[Loop - 1], sizeof(tagSort));
				memcpy(&SortOrder[Loop - 1], &SortOrder[Loop], sizeof(tagSort));
				memcpy(&SortOrder[Loop], &Temp, sizeof(tagSort));
		
				Changed	= TRUE;
			}
		}
	}

	return	ERR_NOERROR;
}

DWORD TScene3D::RenderSortOrder()
{
	for (DWORD Loop = 0; Loop < CurrentSort; Loop++)
	{
		switch (SortOrder[Loop].Type)
		{
			case TYPE_MESH:
				if (Meshes[SortOrder[Loop].Index] != NULL)
				{
					Meshes[SortOrder[Loop].Index]->Render(Cameras[CurrentCamera]);
				}

				break;

			case TYPE_CHARACTER:
				if (Characters[SortOrder[Loop].Index] != NULL)
				{
					Characters[SortOrder[Loop].Index]->Render(Cameras[CurrentCamera]);
				}

				break;
		}
	}

	return	ERR_NOERROR;
}

UINT TScene3D::GetTexture(DWORD TextureID)
{
	if (TextureID >= NumTextures)
	{
		return	-1;
	}

	return	Textures[TextureID].TextureID;
}

DWORD TScene3D::FindSceneItem(DWORD Type, char *Name)
{
	DWORD	Number;

	switch (Type)
	{
		case TYPE_MESH:
			Number	= NumMeshes;

			break;
	
		case TYPE_CHARACTER:
			Number	= NumCharacters;

			break;
	
		case TYPE_LIGHT:
			Number	= NumLights;

			break;
	}

	for (DWORD Loop = 0; Loop < Number; Loop++)
	{
		switch (Type)
		{
			case TYPE_MESH:
				if (TRUE == Meshes[Loop]->FindName(Name))
				{
					return	Loop;
				}
			
				break;

			case TYPE_CHARACTER:
				if (TRUE == Characters[Loop]->FindName(Name))
				{
					return	Loop;
				}
			
				break;

			case TYPE_LIGHT:
				if (TRUE == Lights[Loop]->FindName(Name))
				{
					return	Loop;
				}
			
				break;
		}
	}
		
	return	INVALID;
}

DWORD TScene3D::AddCamera(TVertex *Position, TVertex *LookAt, TVertex *UpVector)
{
	for (DWORD Loop = 0; Loop < MAXCAMERAS; Loop++)
	{
		if (NULL == Cameras[Loop])
		{
			break;
		}
	}

	if (Loop >= MAXCAMERAS)
	{
		return	INVALID;
	}

	Cameras[Loop]	= new TCamera;

	if (NULL == Cameras[Loop])
	{
		return	ERR_MEMORYERROR;
	}

	Cameras[Loop]->SetPosition(Position);
	Cameras[Loop]->SetLookAt(LookAt);
	Cameras[Loop]->SetUpVector(UpVector);

	return	Loop;
}

DWORD TScene3D::SetCurrentCamera(DWORD Camera)
{
	CurrentCamera	= Camera;

	return	ERR_NOERROR;
}

DWORD TScene3D::SetCurrentCamera(char *Name)
{
	for (DWORD Loop = 0; Loop < NumCameras; Loop++)
	{
		if (!strcmp(Cameras[Loop]->GetName(), Name))
		{
			CurrentCamera	= Loop;
		
			break;
		}
	}

	return	ERR_NOERROR;
}

DWORD TScene3D::RotateCurrentCamera(float Angle, float x, float y, float z)
{
	Cameras[CurrentCamera]->Rotate(Angle, x, y, z);
	Cameras[CurrentCamera]->Render();
	ResetFrustum();

	return	ERR_NOERROR;
}

DWORD TScene3D::MoveCurrentCamera(float Distance, BOOL Up)
{
	Cameras[CurrentCamera]->Move(Distance, Up);
	Cameras[CurrentCamera]->Render();
	ResetFrustum();

	return	ERR_NOERROR;
}

DWORD TScene3D::GetCameraPosition(TVertex *Position)
{
	Cameras[CurrentCamera]->GetPosition(Position);

	return	ERR_NOERROR;
}

DWORD TScene3D::SetCameraPosition(TVertex *Position)
{
	Cameras[CurrentCamera]->SetPosition(Position);
	Cameras[CurrentCamera]->Render();
	ResetFrustum();

	return	ERR_NOERROR;
}

DWORD TScene3D::GetCameraLookAt(TVertex *LookAt)
{
	Cameras[CurrentCamera]->GetLookAt(LookAt);

	return	ERR_NOERROR;
}

DWORD TScene3D::SetCameraLookAt(TVertex *LookAt)
{
	Cameras[CurrentCamera]->SetLookAt(LookAt);
	Cameras[CurrentCamera]->Render();
	ResetFrustum();

	return	ERR_NOERROR;
}

DWORD TScene3D::GetCameraUpVector(TVertex *UpVector)
{
	Cameras[CurrentCamera]->GetUpVector(UpVector);

	return	ERR_NOERROR;
}

DWORD TScene3D::SetCameraUpVector(TVertex *UpVector)
{
	Cameras[CurrentCamera]->SetUpVector(UpVector);
	ResetFrustum();
	
	return	ERR_NOERROR;
}

DWORD TScene3D::SetCameraEditMode(BOOL EditMode)
{
	Cameras[CurrentCamera]->SetEditMode(EditMode);

	return	ERR_NOERROR;
}

TFrustum *TScene3D::GetFrustum()
{
	return	Frustum;
}

DWORD TScene3D::ResetFrustum()
{
	Frustum->Initialize();

	CameraMoved	= TRUE;

	return	ERR_NOERROR;
}

DWORD TScene3D::EnableAnimation(DWORD Type, DWORD Mesh)
{
	switch (Type)
	{
		case TYPE_MESH:
			return	Meshes[Mesh]->EnableAnimation();

		case TYPE_CHARACTER:
			return	Characters[Mesh]->EnableAnimation();

		case TYPE_LIGHT:
			return	Lights[Mesh]->EnableAnimation();
	}

	return	ERR_NOERROR;
}

DWORD TScene3D::DisableAnimation(DWORD Type, DWORD Mesh)
{
	switch (Type)
	{
		case TYPE_MESH:
			return	Meshes[Mesh]->DisableAnimation();

		case TYPE_CHARACTER:
			return	Characters[Mesh]->DisableAnimation();

		case TYPE_LIGHT:
			return	Lights[Mesh]->DisableAnimation();
	}

	return	ERR_NOERROR;
}

DWORD TScene3D::GetAnimationLength(DWORD Type, DWORD Mesh)
{
	switch (Type)
	{
		case TYPE_MESH:
			return	Meshes[Mesh]->GetAnimationLength();

		case TYPE_LIGHT:
			return	Lights[Mesh]->GetAnimationLength();
	}

	return	0;
}
	    
DWORD TScene3D::SetAnimation(DWORD Mesh, DWORD Part, char *Name)
{
	return	Characters[Mesh]->SetAnimation(Part, Name);
}

DWORD TScene3D::SetAnimationFrames(DWORD Type, DWORD Mesh, DWORD NewStartFrame, DWORD NewEndFrame)
{
	switch (Type)
	{
		case TYPE_MESH:
			return	Meshes[Mesh]->SetAnimationFrames(NewStartFrame, NewEndFrame);

		case TYPE_LIGHT:
			return	Lights[Mesh]->SetAnimationFrames(NewStartFrame, NewEndFrame);
	}

	return	ERR_NOERROR;
}

DWORD TScene3D::SetAnimationType(DWORD Type, DWORD Mesh, DWORD NewType)
{
	switch (Type)
	{
		case TYPE_MESH:
			return	Meshes[Mesh]->SetAnimationType(NewType);

		case TYPE_CHARACTER:
			return	Characters[Mesh]->SetAnimationType(NewType);

		case TYPE_LIGHT:
			return	Lights[Mesh]->SetAnimationType(NewType);
	}

	return	ERR_NOERROR;
}

DWORD TScene3D::SetAnimationSpeed(DWORD Type, DWORD Mesh, float NewRate)
{
	switch (Type)
	{
		case TYPE_MESH:
			return	Meshes[Mesh]->SetAnimationSpeed(NewRate);

		case TYPE_CHARACTER:
			return	Characters[Mesh]->SetAnimationSpeed(NewRate);

		case TYPE_LIGHT:
			return	Lights[Mesh]->SetAnimationSpeed(NewRate);
	}

	return	ERR_NOERROR;
}

DWORD TScene3D::GetPosition(DWORD Type, char *Name, TVertex *Position)
{
	DWORD	Number;

	switch (Type)
	{
		case TYPE_MESH:
			Number	= NumMeshes;

			break;

		case TYPE_CHARACTER:
			Number	= NumCharacters;

			break;

		case TYPE_LIGHT:
			Number	= NumLights;

			break;
	}

	for (DWORD Loop = 0; Loop < Number; Loop++)
	{ 
		BOOL	Found	= FALSE;

		switch (Type)
		{
			case TYPE_MESH:
				Found	= Meshes[Loop]->FindName(Name);

				break;

			case TYPE_CHARACTER:
				Found	= Characters[Loop]->FindName(Name);

				break;

			case TYPE_LIGHT:
				Found	= Lights[Loop]->FindName(Name);

				break;
		}

		if (TRUE == Found)
		{
			switch (Type)
			{
				case TYPE_MESH:
					Meshes[Loop]->GetPosition(Position);

					break;

				case TYPE_CHARACTER:
					Characters[Loop]->GetPosition(Position);
	
					break;

				case TYPE_LIGHT:
					Lights[Loop]->GetPosition(Position);

					break;
			}
		}
	}

	return	ERR_NOERROR;
}

DWORD TScene3D::GetPosition(DWORD Type, DWORD Mesh, TVertex *Position)
{
	switch (Type)
	{
		case TYPE_MESH:
			return	Meshes[Mesh]->GetPosition(Position);

		case TYPE_CHARACTER:
			return	Characters[Mesh]->GetPosition(Position);

		case TYPE_LIGHT:
			return	Lights[Mesh]->GetPosition(Position);
	}

	return	ERR_NOERROR;
}

DWORD TScene3D::GetPosition(DWORD Type, DWORD Index, char *Name, TVertex *Position)
{
	BOOL	Found	= FALSE;

	switch (Type)
	{
		case TYPE_MESH:
			Found	= Meshes[Index]->FindPosition(Name, Position);

			break;

		case TYPE_CHARACTER:
			Found	= Characters[Index]->FindPosition(Name, Position);

			break;

		case TYPE_LIGHT:
			Found	= Lights[Index]->FindPosition(Name, Position);

			break;
	}

	if (FALSE == Found)
	{
		return	ERR_BADPARAMS;
	}

	return	ERR_NOERROR;
}

DWORD TScene3D::SetPosition(DWORD Type, DWORD Mesh, TVertex *Position)
{
	switch (Type)
	{
		case TYPE_MESH:
			return	Meshes[Mesh]->SetPosition(Position);

		case TYPE_CHARACTER:
			return	Characters[Mesh]->SetPosition(Position);

		case TYPE_LIGHT:
			return	Lights[Mesh]->SetPosition(Position);
	}

	return	ERR_NOERROR;
}

DWORD TScene3D::GetRotation(DWORD Type, DWORD Mesh, TVertex *Rotation)
{
	switch (Type)
	{
		case TYPE_MESH:
			return	Meshes[Mesh]->GetRotation(Rotation);

		case TYPE_CHARACTER:
			return	Characters[Mesh]->GetRotation(Rotation);

		case TYPE_LIGHT:
			return	Lights[Mesh]->GetRotation(Rotation);
	}

	return	ERR_NOERROR;
}

DWORD TScene3D::SetRotation(DWORD Type, DWORD Mesh, TVertex *Rotation)
{
	switch (Type)
	{
		case TYPE_MESH:
			return	Meshes[Mesh]->SetRotation(Rotation);

		case TYPE_CHARACTER:
			return	Characters[Mesh]->SetRotation(Rotation);

		case TYPE_LIGHT:
			return	Lights[Mesh]->SetRotation(Rotation);
	}

	return	ERR_NOERROR;
}

DWORD TScene3D::SetCharacterPath(DWORD Mesh, char *Start, char *End, char *LowerAnimation, 
	char *UpperAnimation, DWORD EndAnimationFlag, float Speed, DWORD Landscape)
{
	TVertex		StartPosition;
	TVertex		EndPosition;
	TVertex		Angle;

	memset(&StartPosition, 0, sizeof(TVertex));
	memset(&EndPosition, 0, sizeof(TVertex));

	for (DWORD Loop = 0; Loop < NumMeshes; Loop++)
	{
		if (TRUE == Meshes[Loop]->FindName(Start))
		{
			Meshes[Loop]->GetPosition(&StartPosition);
		}

		if (TRUE == Meshes[Loop]->FindName(End))
		{
			Meshes[Loop]->GetPosition(&EndPosition);
		}
	}
	
	Paths[Mesh].LandscapeID	= Landscape;
	Paths[Mesh].Current.x	= StartPosition.x;
	Paths[Mesh].Current.y	= StartPosition.y;
	Paths[Mesh].Current.z	= StartPosition.z;
	
	Paths[Mesh].Delta.x		= EndPosition.x - StartPosition.x;
	Paths[Mesh].Delta.y		= EndPosition.y - StartPosition.y;
	Paths[Mesh].Delta.z		= EndPosition.z - StartPosition.z;

	float	Distance	= (sqrtf(Paths[Mesh].Delta.x * Paths[Mesh].Delta.x +
		Paths[Mesh].Delta.y * Paths[Mesh].Delta.y + Paths[Mesh].Delta.z * Paths[Mesh].Delta.z));
	
	Paths[Mesh].Steps		= abs((DWORD)(Distance / Speed));

	if (Paths[Mesh].Steps != 0)
	{
		Paths[Mesh].Delta.x		/= Paths[Mesh].Steps;
		Paths[Mesh].Delta.y		/= Paths[Mesh].Steps;
		Paths[Mesh].Delta.z		/= Paths[Mesh].Steps;
	}

	Angle.x	= 0.0f;
	
	if (Paths[Mesh].Delta.z != 0.0f)
	{		
		Angle.y	= ToDegrees(atanf(-Paths[Mesh].Delta.x / -Paths[Mesh].Delta.z));
		
		if (Paths[Mesh].Delta.z < 0.0f)
		{
			Angle.y	+= 180.0f;
		}
	}

	else
	{
		Angle.y	= 0.0f;
	}

	Angle.z	= 0.0f;

	Characters[Mesh]->SetRotation(&Angle);

	if (Paths[Mesh].LandscapeID != INVALID)
	{
		float	y;
		TVertex	Height;

		Height.x	= Paths[Mesh].Current.x;
		Height.z	= Paths[Mesh].Current.z;

		GetLandscapeHeight(Paths[Mesh].LandscapeID, &Height);

		y	= Height.y;

		if (fabs(Paths[Mesh].Delta.x) > fabs(Paths[Mesh].Delta.z))
		{
			Height.x	+= (Paths[Mesh].Delta.x / (float)fabs(Paths[Mesh].Delta.x));
		}

		else
		{
			Height.z	+= (Paths[Mesh].Delta.z / (float)fabs(Paths[Mesh].Delta.z));
		}

		GetLandscapeHeight(Paths[Mesh].LandscapeID, &Height);

		if (fabs(Paths[Mesh].Delta.x) > fabs(Paths[Mesh].Delta.z))
		{
			Paths[Mesh].Delta.y		= (Height.y - y) * (float)fabs(Paths[Mesh].Delta.x);
		}

		else
		{
			Paths[Mesh].Delta.y		= (Height.y - y) * (float)fabs(Paths[Mesh].Delta.z);
		}

		Paths[Mesh].Current.y		= y + YOFFSET * MD3SCALE;
	}

	strcpy(Paths[Mesh].LowerAnimation, LowerAnimation);
	strcpy(Paths[Mesh].UpperAnimation, UpperAnimation);
	Paths[Mesh].AnimationFlag	= EndAnimationFlag;

	return	ERR_NOERROR;
}

DWORD TScene3D::AnimateCharacter(DWORD Mesh)
{
	Paths[Mesh].Enabled	= TRUE;

	return	ERR_NOERROR;
}

DWORD TScene3D::StopCharacter(DWORD Mesh)
{
	Paths[Mesh].Enabled	= FALSE;

	return	ERR_NOERROR;
}

DWORD TScene3D::AddObject(char *MeshName, char *Filename, TVertex *NewPosition, 
	TVertex *NewRotation, DWORD *ObjectID)
{
	BYTE	*Buffer;
	BYTE	*BufferPtr;
	DWORD	MeshType;
	DWORD	NewMeshes;
	DWORD	TextureID;

	DWORD	MeshID	= GetNextMesh();

	if (INVALID == MeshID)
	{
		return	ERR_MEMORYERROR;
	}

	FILE	*Handle	= fopen(Filename, "rb");

	if (NULL == Handle)
	{
		return	ERR_FILEERROR;
	}

	fseek(Handle, 0, SEEK_END);
	
	DWORD	Length	= ftell(Handle);

	rewind(Handle);

	Buffer	= new BYTE[Length];

	if (NULL == Buffer)
	{
		fclose(Handle);

		return	ERR_MEMORYERROR;
	}

	fread(Buffer, Length, 1, Handle);

	fclose(Handle);

	BufferPtr	= Buffer;

	memcpy(&NewMeshes, BufferPtr, sizeof(DWORD));
	BufferPtr	+= sizeof(DWORD);

	Meshes[MeshID]	= (TBaseMesh3D *)new TObjectMesh3D;

	if (NULL == Meshes[MeshID])
	{
		return	ERR_MEMORYERROR;
	}

	((TObjectMesh3D *)Meshes[MeshID])->Initialize(MeshName, NewMeshes);

	for (DWORD Loop = 0; Loop < NewMeshes; Loop++)
	{
		memcpy(&MeshType, BufferPtr, sizeof(DWORD));
		BufferPtr	+= sizeof(DWORD);
		
		FindTexture(Filename, &TextureID);

		((TObjectMesh3D *)Meshes[MeshID])->Add(MeshType, Loop, &BufferPtr, TextureID);
	}

	ReadTextures(Filename, &BufferPtr, FALSE);

	Meshes[MeshID]->SetPosition(NewPosition);

	if (NewRotation != NULL)
	{
		Meshes[MeshID]->SetRotation(NewRotation);
	}

	if (ObjectID != NULL)
	{
		*ObjectID	= MeshID;
	}

	delete [] Buffer;

	CameraMoved	= TRUE;

	return	ERR_NOERROR;
}

DWORD TScene3D::AddCharacter(char *MeshName, char *Filename, TVertex *NewPosition, 
	TVertex *NewRotation, DWORD *ObjectID)
{
	LoadCharacter(Filename, MeshName, ObjectID);
	Characters[*ObjectID]->SetPosition(NewPosition);
	
	if (NewRotation != NULL)
	{
		Characters[*ObjectID]->SetRotation(NewRotation);
	}

	CameraMoved	= TRUE;

	return	ERR_NOERROR;
}

DWORD TScene3D::RemoveObject(DWORD Type, DWORD Index)
{
	switch (Type)
	{
		case TYPE_MESH:
			if (Index < NumMeshes)
			{
				delete	Meshes[Index];
				
				Meshes[Index]	= NULL;

				if (NumMeshes - 1 == Index && NumMeshes > 0)
				{
					NumMeshes--;
				}
			}

			break;

		case TYPE_CHARACTER:
			if (Index < NumCharacters)
			{
				delete	Characters[Index];
				Characters[Index]	= NULL;

				if (NumCharacters - 1 == Index && NumCharacters > 0)
				{
					NumCharacters--;
				}
			}

			break;
	}

	CameraMoved	= TRUE;
	
	return	ERR_NOERROR;
}

DWORD TScene3D::FindObject(DWORD *Type, DWORD *Index, DWORD *Part, TVertex *NewPosition)
{
	for (DWORD Loop = 0; Loop < NumMeshes; Loop++)
	{
		if (NULL == Meshes[Loop])
		{
			continue;
		}
		
		if (TRUE == Meshes[Loop]->PointInMesh(NewPosition))
		{
			*Type	= TYPE_MESH;
			*Index	= Loop;
			*Part	= 0;

			return	ERR_NOERROR;
		}
	}

	for (Loop = 0; Loop < NumCharacters; Loop++)
	{
		if (NULL == Characters[Loop])
		{
			continue;
		}
		
		for (DWORD InnerLoop = 0; InnerLoop < MAXPARTS; InnerLoop++)
		{
			if (TRUE == Characters[Loop]->PointInMesh(NewPosition, Part))
			{
				*Type	= TYPE_CHARACTER;
				*Index	= Loop;

				return	ERR_NOERROR;
			}
		}
	}

	*Type	= INVALID;
	*Index	= INVALID;
	*Part	= INVALID;
		
	return	ERR_NOERROR;
}

DWORD TScene3D::SelectObject(DWORD Type, DWORD Index)
{
	switch (Type)
	{
		case TYPE_MESH:
			if (Meshes[Index] != NULL)
			{
				Meshes[Index]->Select();
			}

			break;

		case TYPE_CHARACTER:
			if (Characters[Index] != NULL)
			{
				Characters[Index]->Select();
			}

			break;
	}

	CameraMoved	= TRUE;

	return	ERR_NOERROR;
}

DWORD TScene3D::DeselectObject(DWORD Type, DWORD Index)
{
	switch (Type)
	{
		case TYPE_MESH:
			if (Meshes[Index] != NULL)
			{
				Meshes[Index]->Deselect();
			}

			break;

		case TYPE_CHARACTER:
			if (Characters[Index] != NULL)
			{
				Characters[Index]->Deselect();
			}

			break;
	}

	CameraMoved	= TRUE;

	return	ERR_NOERROR;
}

DWORD TScene3D::AddLandscape(char *MeshName, char *HeightMap, TVertex *NewPosition, DWORD Width, 
	DWORD Height, char *Texture, char *Detail, float Scale, float GroundScale, float DetailScale,
	DWORD *ObjectID)
{
	TVertex	Position;
	UINT	GroundTexture;
	UINT	DetailTexture;
	
	DWORD	MeshID	= GetNextMesh();

	if (INVALID == MeshID)
	{
		return	ERR_MEMORYERROR;
	}
	
	Meshes[MeshID]	= (TMesh3D *)new TGroundMesh3D;
	Meshes[MeshID]->Initialize();

	if (NULL == Meshes[MeshID])
	{
		return	ERR_MEMORYERROR;
	}

	GroundTexture	= NumTextures;
	LoadTexture(Texture, FALSE);
	
	DetailTexture	= NumTextures;
	LoadTexture(Detail, FALSE);

	((TGroundMesh3D *)Meshes[MeshID])->Add(HeightMap, NewPosition, Width, Height, 
		GroundTexture, DetailTexture, Scale, GroundScale, DetailScale);
	((TGroundMesh3D *)Meshes[MeshID])->SetName(MeshName);

	Position.x	= 0.0f;
	Position.y	= 0.0f;
	Position.z	= 0.0f;

	Meshes[MeshID]->SetPosition(&Position);

	if (ObjectID != NULL)
	{
		*ObjectID	= MeshID;
	}

	CameraMoved	= TRUE;

	return	ERR_NOERROR;
}

DWORD TScene3D::AddWater(char *MeshName, TVertex *NewPosition, DWORD Width, DWORD Height, 
	UINT TextureID, float Scale, float WaterScale)
{
	TVertex	Position;

	DWORD	MeshID	= GetNextMesh();

	if (INVALID == MeshID)
	{
		return	ERR_MEMORYERROR;
	}
	
	Meshes[MeshID]	= (TMesh3D *)new TWaterMesh3D;
	Meshes[MeshID]->Initialize();

	if (NULL == Meshes[MeshID])
	{
		return	ERR_MEMORYERROR;
	}
	
	((TWaterMesh3D *)Meshes[MeshID])->Add(NewPosition, Width, Height, 
		TextureID, Scale, WaterScale);
	((TWaterMesh3D *)Meshes[MeshID])->SetName(MeshName);

	Position.x	= 0.0f;
	Position.y	= 0.0f;
	Position.z	= 0.0f;

	Meshes[MeshID]->SetPosition(&Position);

	return	ERR_NOERROR;
}

DWORD TScene3D::AddTree(char *MeshName, TVertex *Position, float Width, float Height, 
	UINT TextureID)
{
	if (TextureID >= NumTextures)
	{
		return	ERR_BADPARAMS;
	}

	DWORD	MeshID	= GetNextMesh();

	if (INVALID == MeshID)
	{
		return	ERR_MEMORYERROR;
	}
	
	Meshes[MeshID]	= (TBaseMesh3D *)new TBillboardMesh3D;
	Meshes[MeshID]->Initialize();

	if (NULL == Meshes[MeshID])
	{
		return	ERR_MEMORYERROR;
	}

	((TBillboardMesh3D *)Meshes[MeshID])->Add(MeshName, Position, Width, Height, TextureID);

	return	ERR_NOERROR;
}

DWORD TScene3D::GetLandscapeHeight(DWORD Mesh, TVertex *Height)
{
	((TGroundMesh3D *)Meshes[Mesh])->GetHeight(Height);

	return	ERR_NOERROR;
}

DWORD TScene3D::GetWaterHeight(DWORD Mesh, TVertex *Height)
{
	((TWaterMesh3D *)Meshes[Mesh])->GetHeight(Height);

	return	ERR_NOERROR;
}

DWORD TScene3D::ReadTextures(char *Filename, BYTE **Buffer, BOOL Clamp)
{
	DWORD	ReadTextures;

	for (DWORD Loop = 0; Loop < NumTextures; Loop++)
	{
		if (!strcmp(Filename, Textures[Loop].Name))
		{
			return	ERR_NOERROR;
		}
	}

	memcpy(&ReadTextures, *Buffer, sizeof(DWORD));
	*Buffer	+= sizeof(DWORD);

	for (Loop = 0; Loop < ReadTextures; Loop++)
	{
		DWORD	Width;
		DWORD	Height;
		DWORD	Alpha;
		DWORD	Modulate;

		memcpy(&Width, *Buffer, sizeof(DWORD));
		*Buffer	+= sizeof(DWORD);
	
		memcpy(&Height, *Buffer, sizeof(DWORD));
		*Buffer	+= sizeof(DWORD);

		memcpy(&Alpha, *Buffer, sizeof(DWORD));
		*Buffer	+= sizeof(DWORD);

		memcpy(&Modulate, *Buffer, sizeof(DWORD));
		*Buffer	+= sizeof(DWORD);

		glPixelStorei(GL_UNPACK_ALIGNMENT, 1);
		glGenTextures(1, &Textures[Loop + NumTextures].TextureID);

		char	String[128];
					
		sprintf(String, "Texture %d Loop %d Width %d Height %d.\n", 
			Textures[Loop + NumTextures].TextureID, Loop, Width, Height);
						
		OutputDebugString(String);

		glBindTexture(GL_TEXTURE_2D, Textures[Loop + NumTextures].TextureID);

		if (TRUE == Alpha)
		{
			gluBuild2DMipmaps(GL_TEXTURE_2D, 4, Width, Height, GL_RGBA, GL_UNSIGNED_BYTE, 
				*Buffer);
		}

		else
		{
			gluBuild2DMipmaps(GL_TEXTURE_2D, 3, Width, Height, GL_RGB, GL_UNSIGNED_BYTE, 
				*Buffer);
		}

		if (TRUE == Alpha)
		{
			*Buffer	+= Width * Height * 4;
		}

		else
		{
			*Buffer	+= Width * Height * 3;
		}

		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST_MIPMAP_NEAREST);
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
		
		if (TRUE == Modulate)
		{
			glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);
		}

		else
		{
			glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_REPLACE);
		}

		if (TRUE == Clamp)
		{
			glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE);
			glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_EDGE);
		}

		else
		{
			glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
			glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);
		}
	}

	strcpy(Textures[NumTextures].Name, Filename);

	NumTextures	+= ReadTextures;

	return	ERR_NOERROR;
}

DWORD TScene3D::LoadTexture(char *Filename, BOOL Clamp, DWORD *TextureID)
{
	BYTE		TGACompare[12];
	BYTE		Header[6];
	DWORD		ImageSize;
	DWORD		Width;
	DWORD		Height;
	DWORD		Bytes;
	BYTE		*Texture;

	if (NumTextures >= MAXTEXTURES)
	{
		return	ERR_MEMORYERROR;
	}

	FILE	*Handle	= fopen(Filename, "rb");

	if (NULL == Handle)
	{
		return	ERR_FILEERROR;
	}
	 
	fread(TGACompare, 1, sizeof(TGACompare), Handle);
	fread(Header, 1, sizeof(Header), Handle);

	Width	= Header[1] * 256 + Header[0];
	Height	= Header[3] * 256 + Header[2];
    
 	if (0 == Width || 0 == Height || (Header[4] != 24 && Header[4] != 32))
	{
		fclose(Handle);
			
		return	ERR_FILEERROR;
	}

	Bytes		= Header[4] / 8;
	ImageSize	= Width * Height * Bytes;

	Texture		= new BYTE[ImageSize];

	if (NULL == Texture)
	{
		fclose(Handle);
				
		return	ERR_MEMORYERROR;
	}

	fread(Texture, 1, ImageSize, Handle);

	for (DWORD InnerLoop = 0; InnerLoop < ImageSize; InnerLoop += Bytes)
	{								
		BYTE	Temp			= Texture[InnerLoop];	
		Texture[InnerLoop]		= Texture[InnerLoop + 2];
		Texture[InnerLoop + 2]	= Temp;					
	}

	fclose(Handle);

	glPixelStorei(GL_UNPACK_ALIGNMENT, 1);
	glGenTextures(1, &Textures[NumTextures].TextureID);

	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR_MIPMAP_NEAREST);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);

	if (TRUE == Clamp)
	{
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE);
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_EDGE);
	}

	else
	{
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);
	}

	glBindTexture(GL_TEXTURE_2D, Textures[NumTextures].TextureID);

	if (4 == Bytes)
	{
		gluBuild2DMipmaps(GL_TEXTURE_2D, 4, Width, Height, GL_RGBA, GL_UNSIGNED_BYTE, Texture);
	}

	else
	{
		gluBuild2DMipmaps(GL_TEXTURE_2D, 3, Width, Height, GL_RGB, GL_UNSIGNED_BYTE, Texture);
	}
	
	if (TextureID != NULL)
	{
		*TextureID	= NumTextures;
	}

	strcpy(Textures[NumTextures].Name, Filename);

	NumTextures++;

	delete [] Texture;

	return	ERR_NOERROR;
}

DWORD TScene3D::RemoveTexture(DWORD TextureID)
{
	if (Textures[TextureID].TextureID != -1)
	{
		glDeleteTextures(1, &Textures[TextureID].TextureID);
	
		Textures[TextureID].TextureID	= -1;
	}
		
	return	ERR_NOERROR;
}

DWORD TScene3D::FindTexture(char *Filename, DWORD *TextureID)
{
	for (DWORD Loop = 0; Loop < NumTextures; Loop++)
	{
		if (!strcmp(Filename, Textures[Loop].Name))
		{
			if (TextureID != NULL)
			{
				*TextureID	= Loop;
			}

			return	ERR_NOERROR;
		}
	}

	if (TextureID != NULL)
	{
		*TextureID	= NumTextures;
	}	

	return	ERR_NOERROR;
}

DWORD TScene3D::GetNextMesh()
{
	DWORD	RetVal	= NumMeshes;

	for (DWORD Loop = 0; Loop < NumMeshes; Loop++)
	{
		if (NULL == Meshes[Loop])
		{
			return	Loop;
		}
	}

	NumMeshes++;

	if (NumMeshes >= MAXMESHES)
	{
		return	INVALID;
	}

	return	RetVal;
}

DWORD TScene3D::GetNextCharacter()
{
	DWORD	RetVal	= NumCharacters;

	for (DWORD Loop = 0; Loop < NumCharacters; Loop++)
	{
		if (NULL == Characters[Loop])
		{
			return	Loop;
		}
	}

	NumCharacters++;

	if (NumCharacters >= MAXMESHES)
	{
		return	INVALID;
	}

	return	RetVal;
}

DWORD TScene3D::CreateOctree()
{
	return	ERR_NOERROR;
}

#ifdef SHOWFRAMERATE
DWORD TScene3D::CalculateFrameRate()
{
	static float	FramesPerSecond	= 0.0f;
	static float	LastTime		= 0.0f;
	static float	FrameTime		= 0.0f;
	float			CurrentTime		= timeGetTime() * 0.001f;				
	char			String[128];

 	FrameInterval	= CurrentTime - FrameTime;
	FrameTime		= CurrentTime;

    FramesPerSecond++;

	if (CurrentTime - LastTime > 1.0f)
	{
		LastTime	= CurrentTime;
		
		sprintf(String, "Frame Per Second: %d", int(FramesPerSecond));

		if (Window != NULL)
		{
			SetWindowText(Window->GetHwnd(), String);
		}

		FramesPerSecond = 0;
    }

	return	ERR_NOERROR;
}
#endif